//
// Tizen C++ SDK
// Copyright (c) 2012-2013 Samsung Electronics Co., Ltd.
//
// Licensed under the Flora License, Version 1.1 (the License);
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://floralicense.org/license/
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an AS IS BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

#include <math.h>
#include "Box2dTestBed.h"

using namespace Tizen::Base;
using namespace Tizen::Base::Runtime;
using namespace Tizen::Graphics;
using namespace Tizen::App;
using namespace Tizen::System;
using namespace Tizen::Ui;
using namespace Tizen::Ui::Controls;
using namespace Tizen::Graphics::Opengl;

const int TIME_OUT = 10;

#define DISPLAY_FPS

class GlesForm: public Tizen::Ui::Controls::Form {
public:
	GlesForm(Box2dTestBed* pApp) :
			__pApp(pApp) {
	}

	virtual ~GlesForm(void) {
	}

	virtual result OnDraw(void) {
		return E_SUCCESS;
	}

private:
	Box2dTestBed* __pApp;
};

namespace // unnamed
{
#define glClearColorEx glClearColor
#define glOrtho 	glOrthof
}

Box2dTestBed::Box2dTestBed(void)
	: __eglDisplay(EGL_NO_DISPLAY)
	, __eglSurface(EGL_NO_SURFACE)
	, __eglConfig(null)
	, __eglContext(EGL_NO_CONTEXT)
	, __pTimer(null)
	, __pForm(null)
	, __pTestTitle(null)
	, __pFPSLabel(null)
	, __pHelpLabel(null)
	, __hints(L"")
	, __prevX(0)
	, __prevY(0)
	, __testNumber (0)
	, __isBodyAtPoint(false)
	, __pinballArmsUp(false)
	, __pTest(null)
	, __viewZoom(1.0f)
{
}

Box2dTestBed::~Box2dTestBed(void)
{
}

void
Box2dTestBed::Cleanup(void)
{
	if (__pTimer != null) {
		__pTimer->Cancel();
		delete __pTimer;
		__pTimer = null;
	}

	DestroyGL();
}

Application*
Box2dTestBed::CreateInstance(void)
{
	return new (std::nothrow) Box2dTestBed();
}

bool
Box2dTestBed::OnAppInitializing(AppRegistry& appRegistry)
{
	Timer* pTempTimer = null;
	result r = E_SUCCESS;
	Frame* pAppFrame = new (std::nothrow) Frame();
	TryReturn(pAppFrame != null, E_FAILURE, "[Box2dTestBed] Generating a frame failed.");

	r = pAppFrame->Construct();
	TryCatch(!IsFailed(r), , "[Box2dTestBed] pAppFrame->Construct() failed.");

	this->AddFrame(*pAppFrame);

	__pForm = new (std::nothrow) GlesForm(this);
	TryCatch(__pForm != null, , "[Box2dTestBed] Allocation of GlesForm failed.");

	r = __pForm->Construct(FORM_STYLE_NORMAL);
	TryCatch(!IsFailed(r), delete __pForm, "[Box2dTestBed] __pForm->Construct(FORM_STYLE_NORMAL) failed.");

	r = GetAppFrame()->GetFrame()->AddControl(__pForm);
	TryCatch(!IsFailed(r), delete __pForm, "[Box2dTestBed] GetAppFrame()->GetFrame()->AddControl(__pForm) failed.");

	GetAppFrame()->GetFrame()->SetOrientation(ORIENTATION_LANDSCAPE);

	__pForm->AddKeyEventListener(*this);
	__pForm->AddTouchEventListener(*this);

	TryCatch(InitEGL(), , "[Box2dTestBed] GlesCube::InitEGL() failed.");

	TryCatch(InitGL(), , "[Box2dTestBed] GlesCube::InitGL() failed.");

	pTempTimer = new (std::nothrow) Timer;
	TryCatch(pTempTimer != null, , "[Box2dTestBed] Failed to allocate memory.");

	r = pTempTimer->Construct(*this);
	TryCatch(!IsFailed(r), , "[Box2dTestBed] pTempTimer->Construct(*this) failed.");

	__pTimer  = pTempTimer;
	pTempTimer = 0;

	UICreating();

    SetTest(__testNumber);

	return true;

CATCH:
	delete pTempTimer;
	delete pAppFrame;

	AppLog("[Box2dTestBed] Box2dTestBed::OnAppInitializing eglError : %#x\n"
			"[Box2dTestBed] Box2dTestBed::OnAppInitializing glError : %#x\n"
			"[Box2dTestBed] Box2dTestBed::OnAppInitializing VENDOR : %s\n"
			"[Box2dTestBed] Box2dTestBed::OnAppInitializing GL_RENDERER : %s\n"
			"[Box2dTestBed] Box2dTestBed::OnAppInitializing GL_VERSION : %s\n ",
			static_cast<unsigned int>(eglGetError()),
			static_cast<unsigned int>(glGetError()),
			glGetString(GL_VENDOR),
			glGetString(GL_RENDERER),
			glGetString(GL_VERSION));

	Cleanup();

	return false;
}

bool
Box2dTestBed::OnAppTerminating(AppRegistry& appRegistry, bool forcedTermination)
{
	Cleanup();
	return true;
}

void
Box2dTestBed::OnForeground(void)
{
	if (__pTimer != null) {
		__pTimer->Start(TIME_OUT);
	}
}

void
Box2dTestBed::OnBackground(void)
{
	if (__pTimer != null) {
		__pTimer->Cancel();
	}
}

void
Box2dTestBed::OnLowMemory(void)
{
}

void
Box2dTestBed::OnBatteryLevelChanged(BatteryLevel batteryLevel)
{
}

void
Box2dTestBed::OnTimerExpired(Timer& timer)
{
	if (__pTimer == null) {
		return;
	}

	__pTimer->Start(TIME_OUT);

	if (!Draw()) {
		AppLog("[Box2dTestBed] Box2dTestBed::Draw() failed.");
	}
}

void
Box2dTestBed::OnKeyPressed(const Control& source, Tizen::Ui::KeyCode keyCode)
{
}

void
Box2dTestBed::OnKeyReleased(const Control& source, Tizen::Ui::KeyCode keyCode)
{
	if (keyCode == Tizen::Ui::KEY_BACK)
	{
		Terminate();
	}
}

void
Box2dTestBed::OnKeyLongPressed(const Control& source, Tizen::Ui::KeyCode keyCode)
{
}

void
Box2dTestBed::OnTouchCanceled (const Control &source, const Point &currentPosition, const TouchEventInfo &touchInfo)
{

}

void
Box2dTestBed::OnTouchFocusIn (const Control &source, const Point &currentPosition, const TouchEventInfo &touchInfo)
{
}

void
Box2dTestBed::OnTouchFocusOut (const Control &source, const Point &currentPosition, const TouchEventInfo &touchInfo)
{
}

void
Box2dTestBed::OnTouchMoved (const Control &source, const Point &currentPosition, const TouchEventInfo &touchInfo)
{
	if (__isBodyAtPoint) {
		b2Vec2 p = ConvertScreenToWorld(currentPosition.x, currentPosition.y);
		__pTest->MouseMove(p);
	} else {
		int deltaX = currentPosition.x - __prevX;
		int deltaY = currentPosition.y - __prevY;

		__settings.viewCenter.x -= (float) deltaX / T_WIDTH * 36.0;
		__settings.viewCenter.y += (float) deltaY / T_HEIGHT * 24.0;

		Resize();
	}

	__prevX = currentPosition.x;
	__prevY = currentPosition.y;
}

void
Box2dTestBed::OnTouchPressed (const Control &source, const Point &currentPosition, const TouchEventInfo &touchInfo)
{
	__prevX = currentPosition.x;
	__prevY = currentPosition.y;

	b2Vec2 p = ConvertScreenToWorld(currentPosition.x, currentPosition.y);
	__isBodyAtPoint = __pTest->MouseDown(p);
}

void
Box2dTestBed::OnTouchReleased (const Control &source, const Point &currentPosition, const TouchEventInfo &touchInfo)
{
	b2Vec2 p = ConvertScreenToWorld(currentPosition.x, currentPosition.y);
	__pTest->MouseUp(p);
	__isBodyAtPoint = false;
}

void
Box2dTestBed::OnActionPerformed(const Tizen::Ui::Control& source,
		int actionId) {

	String str = source.GetName();

	if (str != null) {
		switch (actionId) {
		case ID_A_OPTION: {
			if (__testNumber == 21) {
				if (__pinballArmsUp)
					__pTest->KeyboardUp('a');
				else
					__pTest->Keyboard('a');

				__pinballArmsUp = !__pinballArmsUp;
			} else {
				__pTest->Keyboard('a');
			}
		}
			break;
		case ID_C_OPTION: {
			if ((__testNumber == 17) | (__testNumber == 25)
					| (__testNumber == 29)) {
				int r = rand() % 5 + 1;
				__pTest->Keyboard('0' + r);
			} else {
				__pTest->Keyboard('c');
			}
		}
			break;
		default:
			__pTest->Keyboard(str[0]);
			break;
		}
	} else {
		switch (actionId) {
		case ID_PREV_BUTTON: {
			if (__testNumber == 0)
				__testNumber = 43;
			else
				__testNumber--;
			SetTest(__testNumber);
		}
			break;
		case ID_RESET_BUTTON:
			SetTest(__testNumber);
			break;
		case ID_NEXT_BUTTON: {
			if (__testNumber == 43)
				__testNumber = 0;
			else
				__testNumber++;
			SetTest(__testNumber);
		}
			break;
		case ID_PAUSE_BUTTON: {
			if (!__settings.pause)
				__settings.pause = 1;
			else
				__settings.pause = 0;
		}
			break;
		case ID_EXIT_BUTTON:
			Terminate();
			break;
		case ID_KEYBOARD: {
			if (__pKeyboardPanel != null)
				__pKeyboardPanel->SetShowState(!__pKeyboardPanel->IsVisible());
		}
			break;
		case ID_BOMB_BUTTON:
			__pTest->LaunchBomb();
			break;
		case ID_ZOOM_PLUS_BUTTON:
			__viewZoom = b2Max(0.9f * __viewZoom, 0.02f);
			Resize();
			break;
		case ID_ZOOM_MINUS_BUTTON:
			__viewZoom = b2Min(1.1f * __viewZoom, 20.0f);
			Resize();
			break;
		default:
			break;
		}
	}
}

bool
Box2dTestBed::InitEGL(void)
{
	EGLint numConfigs = 1;

	EGLint eglConfigList[] =
	{
		EGL_RED_SIZE,	8,
		EGL_GREEN_SIZE,	8,
		EGL_BLUE_SIZE,	8,
		EGL_ALPHA_SIZE,	0,
		EGL_DEPTH_SIZE, 8,
		EGL_SURFACE_TYPE, EGL_WINDOW_BIT,
		EGL_RENDERABLE_TYPE, EGL_OPENGL_ES_BIT,
		EGL_NONE
	};

	EGLint eglContextList[] =
	{
		EGL_CONTEXT_CLIENT_VERSION, 1,
		EGL_NONE
	};

	eglBindAPI(EGL_OPENGL_ES_API);

	__eglDisplay = eglGetDisplay((EGLNativeDisplayType)EGL_DEFAULT_DISPLAY);
	TryCatch(__eglDisplay != EGL_NO_DISPLAY, , "[Box2dTestBed] eglGetDisplay() failed.");

	TryCatch(!(eglInitialize(__eglDisplay, null, null) == EGL_FALSE || eglGetError() != EGL_SUCCESS), , "[Box2dTestBed] eglInitialize() failed.");

	TryCatch(!(eglChooseConfig(__eglDisplay, eglConfigList, &__eglConfig, 1, &numConfigs) == EGL_FALSE ||
			eglGetError() != EGL_SUCCESS), , "[Box2dTestBed] eglChooseConfig() failed.");

	TryCatch(numConfigs, , "[Box2dTestBed] eglChooseConfig() failed. because of matching config doesn't exist");

	__eglSurface = eglCreateWindowSurface(__eglDisplay, __eglConfig, (EGLNativeWindowType)__pForm, null);
	TryCatch(!(__eglSurface == EGL_NO_SURFACE || eglGetError() != EGL_SUCCESS), , "[Box2dTestBed] eglCreateWindowSurface() failed.");

	__eglContext = eglCreateContext(__eglDisplay, __eglConfig, EGL_NO_CONTEXT, eglContextList);
	TryCatch(!(__eglContext == EGL_NO_CONTEXT || eglGetError() != EGL_SUCCESS), , "[Box2dTestBed] eglCreateContext() failed.");

	TryCatch(!(eglMakeCurrent(__eglDisplay, __eglSurface, __eglSurface, __eglContext) == EGL_FALSE ||
			eglGetError() != EGL_SUCCESS), , "[Box2dTestBed] eglMakeCurrent() failed.");

	return true;

CATCH:
	{
		AppLog("[Box2dTestBed] Box2dTestBed can run on systems which supports OpenGL ES(R) 1.1.");
		AppLog("[Box2dTestBed] When Box2dTestBed does not correctly execute, there are a few reasons.");
		AppLog("[Box2dTestBed]    1. The current device(real-target or emulator) does not support OpenGL ES(R) 1.1.\n"
				" Check the Release Notes.");
		AppLog("[Box2dTestBed]    2. The system running on emulator cannot support OpenGL(R) 1.5 or later.\n"
				" Try with other system.");
		AppLog("[Box2dTestBed]    3. The system running on emulator does not maintain the latest graphics driver.\n"
				" Update the graphics driver.");
	}

	DestroyGL();

	return false;
}

bool
Box2dTestBed::InitGL(void)
{
	int x, y, width, height;
	__pForm->GetBounds(x, y, width, height);

	glShadeModel(GL_SMOOTH);

	glViewport(0, 0, width, height);

	glEnable(GL_CULL_FACE);
	glCullFace(GL_BACK);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);

	glClearColorEx(0.0f, 0.0f, 0.0f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	return true;
}

void
Box2dTestBed::DestroyGL(void)
{
	if (__eglDisplay != EGL_NO_DISPLAY) {
		eglMakeCurrent(__eglDisplay, null, null, null);

		if (__eglContext != EGL_NO_CONTEXT) {
			eglDestroyContext(__eglDisplay, __eglContext);
			__eglContext = EGL_NO_CONTEXT;
		}

		if (__eglSurface != EGL_NO_SURFACE) {
			eglDestroySurface(__eglDisplay, __eglSurface);
			__eglSurface = EGL_NO_SURFACE;
		}

		eglTerminate(__eglDisplay);
		__eglDisplay = EGL_NO_DISPLAY;
	}

	__eglConfig = null;

	return;
}

bool
Box2dTestBed::Draw(void)
{
	EGLBoolean ret = EGL_FALSE;

	ret = eglMakeCurrent(__eglDisplay, __eglSurface, __eglSurface,
			__eglContext);

	if (ret == EGL_FALSE || eglGetError() != EGL_SUCCESS) {
		AppLog("[Box2dTestBed] eglMakeCurrent() failed.");
		return false;
	}

	struct {
		GLint x, y, width, height;
	} viewPort;

	glGetIntegerv(GL_VIEWPORT, (GLint*) &viewPort);

	{
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

		__hints = L"";
		DrawString(L"");

		glEnableClientState(GL_VERTEX_ARRAY);
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

		b2Vec2 oldCenter = __settings.viewCenter;

		__pTest->Step(&__settings);

		if (oldCenter.x != __settings.viewCenter.x
				|| oldCenter.y != __settings.viewCenter.y) {
			Resize();
		}

		glDisable(GL_BLEND);
		glDisableClientState(GL_VERTEX_ARRAY);
	}

	eglSwapBuffers(__eglDisplay, __eglSurface);

#if defined(DISPLAY_FPS)
	static float fps = 0.0f;
	static float updateInterval = 1000.0f;
	static float timeSinceLastUpdate = 0.0f;
	static float frameCount = 0;
	static long long currentTick;
	static long long lastTick;
	static bool isFirst = true;

	if (isFirst) {
		SystemTime::GetTicks(currentTick);
		lastTick = currentTick;
		isFirst = false;
	}

	frameCount++;
	SystemTime::GetTicks(currentTick);

	float elapsed = currentTick - lastTick;

	lastTick = currentTick;
	timeSinceLastUpdate += elapsed;

	if (timeSinceLastUpdate > updateInterval) {
		if (timeSinceLastUpdate) {
			fps = (frameCount / timeSinceLastUpdate) * 1000.f;
//			AppLog("[Box2dTestBed] FPS: %f frames/sec", fps);
			if (__pFPSLabel != null) {
				__pFPSLabel->SetText(Float::ToString(fps));
				__pFPSLabel->Invalidate(false);
			}

			frameCount = 0;
			timeSinceLastUpdate -= updateInterval;
		}
	}
#endif

	return true;
}

void
Box2dTestBed::SetTest(int number) {
	__pTest = g_testEntries[number].createFcn();

	if (__pTestTitle != null) {
		__pTestTitle->SetText(g_testEntries[number].name);
		__pTestTitle->Invalidate(false);
	}

	if (__settings.viewCenter.x != 0 || __settings.viewCenter.y != 8) {
		__settings.viewCenter.x = 0;
		__settings.viewCenter.y = 8;
		Resize();
	}
}

b2Vec2
Box2dTestBed::ConvertScreenToWorld(int32 x, int32 y) {
	float32 u = x / float32(T_WIDTH);
	float32 v = (T_HEIGHT - y) / float32(T_HEIGHT);

	float32 ratio = float32(T_WIDTH) / float32(T_HEIGHT);
	b2Vec2 extents(ratio * 12.0f, 12.0f);
	extents *= __viewZoom;

	b2Vec2 lower = __settings.viewCenter - extents;
	b2Vec2 upper = __settings.viewCenter + extents;

	b2Vec2 p;
	p.x = (1.0f - u) * lower.x + u * upper.x;
	p.y = (1.0f - v) * lower.y + v * upper.y;

	return p;
}

void
Box2dTestBed::Resize() {
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	float32 ratio = float32(T_WIDTH) / float32(T_HEIGHT);

	b2Vec2 extents(ratio * 12.0f, 12.0f);
	extents *= __viewZoom;

	b2Vec2 lower = __settings.viewCenter - extents;
	b2Vec2 upper = __settings.viewCenter + extents;

	// L/R/B/T
	glOrtho(lower.x, upper.x, lower.y, upper.y, -1, 1);
}

void
Box2dTestBed::DrawString(String text) {
	if (text.Contains(L"1-5"))
		text.Replace(L"1-5", L"'c'");

	if (!text.IsEmpty())
		__hints += text + "\n";

	if (__pHelpLabel != null) {
		__pHelpLabel->SetText(__hints);
		__pHelpLabel->Invalidate(false);
	}
}

void
Box2dTestBed::UICreating(void) {
	// Menu's buttons
	Button* pPrevButton = new Button();
	pPrevButton->Construct(Rectangle(10, 650, 100, 60), L"Prev");
	pPrevButton->SetActionId(ID_PREV_BUTTON);
	pPrevButton->AddActionEventListener(*this);
	pPrevButton->SetTextSize(28);
	__pForm->AddControl(pPrevButton);

	Button* pResetButton = new Button();
	pResetButton->Construct(Rectangle(110, 650, 120, 60), L"Restart");
	pResetButton->SetActionId(ID_RESET_BUTTON);
	pResetButton->AddActionEventListener(*this);
	pResetButton->SetTextSize(28);
	__pForm->AddControl(pResetButton);

	Button* pNextButton = new Button();
	pNextButton->Construct(Rectangle(230, 650, 100, 60), L"Next");
	pNextButton->SetActionId(ID_NEXT_BUTTON);
	pNextButton->AddActionEventListener(*this);
	pNextButton->SetTextSize(28);
	__pForm->AddControl(pNextButton);

	Button* pPauseButton = new Button();
	pPauseButton->Construct(Rectangle(330, 650, 100, 60), L"Pause");
	pPauseButton->SetActionId(ID_PAUSE_BUTTON);
	pPauseButton->AddActionEventListener(*this);
	pPauseButton->SetTextSize(28);
	__pForm->AddControl(pPauseButton);

	Button* pBombButton = new Button();
	pBombButton->Construct(Rectangle(430, 650, 120, 60), L"Bomb!");
	pBombButton->SetActionId(ID_BOMB_BUTTON);
	pBombButton->AddActionEventListener(*this);
	pBombButton->SetTextSize(28);
	__pForm->AddControl(pBombButton);

	Button* pExitButton = new Button();
	pExitButton->Construct(Rectangle(1140, 10, 120, 60), L"Exit");
	pExitButton->SetActionId(ID_EXIT_BUTTON);
	pExitButton->AddActionEventListener(*this);
	__pForm->AddControl(pExitButton);

	Button* pZoomPlusButton = new Button();
	pZoomPlusButton->Construct(Rectangle(1140, 80, 60, 60), L"+");
	pZoomPlusButton->SetActionId(ID_ZOOM_PLUS_BUTTON);
	pZoomPlusButton->AddActionEventListener(*this);
	__pForm->AddControl(pZoomPlusButton);

	Button* pZoomMinusButton = new Button();
	pZoomMinusButton->Construct(Rectangle(1200, 80, 60, 60), L"-");
	pZoomMinusButton->SetActionId(ID_ZOOM_MINUS_BUTTON);
	pZoomMinusButton->AddActionEventListener(*this);
	__pForm->AddControl(pZoomMinusButton);

	Button* pKeyboard = new Button();
	pKeyboard->Construct(Rectangle(10, 10, 120, 60), L"Keys");
	pKeyboard->SetActionId(ID_KEYBOARD);
	pKeyboard->AddActionEventListener(*this);
	__pForm->AddControl(pKeyboard);

	__pTestTitle = new Label();
	__pTestTitle->Construct(Rectangle(520, 660, 520, 50),
			g_testEntries[__testNumber].name);
	__pTestTitle->SetTextConfig(40, LABEL_TEXT_STYLE_ITALIC);
	__pTestTitle->SetTextColor(Color::GetColor(COLOR_ID_GREY));
	__pTestTitle->SetTextVerticalAlignment(ALIGNMENT_MIDDLE);
	__pTestTitle->SetTextHorizontalAlignment(ALIGNMENT_CENTER);
	__pForm->AddControl(__pTestTitle);

	__pFPSLabel = new Label();
	__pFPSLabel->Construct(Rectangle(1060, 660, 200, 50), L"");
	__pFPSLabel->SetTextConfig(30, LABEL_TEXT_STYLE_ITALIC);
	__pFPSLabel->SetTextColor(Color::GetColor(COLOR_ID_GREY));
	__pFPSLabel->SetTextHorizontalAlignment(ALIGNMENT_RIGHT);
	__pForm->AddControl(__pFPSLabel);

	__pHelpLabel = new Label();
	__pHelpLabel->Construct(Rectangle(150, 20, 1000, 120), L"");
	__pHelpLabel->SetTextConfig(24, LABEL_TEXT_STYLE_NORMAL);
	__pHelpLabel->SetTextColor(Color::GetColor(COLOR_ID_GREY));
	__pHelpLabel->SetTextHorizontalAlignment(ALIGNMENT_CENTER);
	__pHelpLabel->SetTextVerticalAlignment(ALIGNMENT_TOP);
	__pForm->AddControl(__pHelpLabel);

	__pKeyboardPanel = new Panel();
	__pKeyboardPanel->Construct(Rectangle(5, 80, 300, 300));
	__pKeyboardPanel->SetShowState(false);

	static const int ROWS_COUNT = 4;
	static const int BUTTONS_COUNT = 14;

	String keyboard[] = {"q", "w", "e", ",", "a", "s", "d", "f", "j", "k", "l", "b", "c", "m"};
	Button* buttons[14];
	int index, i, j;

	for (j = 0; j < ROWS_COUNT; j++) {
		for (i = 0; i < ROWS_COUNT; i++) {
			index = i + ROWS_COUNT * j;
			if (index < BUTTONS_COUNT) {
				buttons[index] = new Button();
				buttons[index]->Construct(Rectangle(10 + 70 * i, 10 + 70 * j, 60, 60), keyboard[index]);
				buttons[index]->SetActionId(201 + index);
				buttons[index]->SetName(keyboard[index]);
				buttons[index]->AddActionEventListener(*this);
			}
		}
	}

	for (i = 0; i < BUTTONS_COUNT; i++)
		__pKeyboardPanel->AddControl(buttons[i]);

	__pForm->AddControl(__pKeyboardPanel);
}
