﻿using System;
using System.Diagnostics;
using System.IO;
using System.Threading;
using System.Collections.Generic;

namespace RungeKutty
{
    internal abstract class RungeKutty
    {
        //-----------Stałe i Tablice-------------------//
        protected const double G = -9.81;
        protected static int N = 100;
        protected const double Tmax = 10;
        protected const double H = 0.001;
        protected double[] Y;
        protected double[] Ynast;
        protected const int Threads = 1;
        protected List<Tuple<int, int>> Limiter;
        //----------------Swobodny Spadek----------------------//
        public abstract void Calculate();
        protected double Throw(int i, double[] y)
        {
            double wynik = 0;
            switch (i)
            {
                case 0:
                    wynik = y[1];
                    break;
                case 1:
                    wynik = -G;
                    break;
                default:
                    throw new SystemException("Zły numer równania");
            }
            return wynik;
        }
        //-----------------Oscylator Harmoniczny-------------------//
        protected double Oscylator(int i, double[] y)
        {
            const double w = 1;
            const double b = 0;

            double wynik = 0;
            switch (i)
            {
                case 0:
                    wynik = y[1];
                    break;
                case 1:
                    wynik = -w * w * y[0] - 2 * b * y[1];
                    break;
                default:
                    throw new SystemException("Zły numer równania");
            }
            return wynik;
        }
        //----------------Oscylatory sprzezone--------------------//
        protected double Osc_Spr(int i, double[] y)
        {
            const int n = 14;
            const double w = 1;
            const double b = 0;
            const double l = 1;

            double wynik = 0;
            if(i % 2 == 0)
            {
                wynik = y[i + 1];
            }
            else
            {
                wynik = -2 * b * y[i];
                if (i > 1) wynik += -w * w * (y[i - 1] - y[i - 3] - l);
                if (i < N - 1) wynik += +w * w * (y[i + 1] - y[i - 1] - l);
            }
            return wynik;
        }
        //-------------Inicjalizacja Tablic---------------------------//
        protected void Initialize()
        {
            Y = new double[N];
            Ynast = new double[N];
            Limiter = new List<Tuple<int, int>>();
            for (var i = 0; i < N; ++i)
            {
                Y[i] = 0;
                Ynast[i] = 0;
            }
            int tmp = 0;
            for (int i=0; i<N; i += (N/Threads))
            {
                tmp += (N / Threads);
                Limiter.Add(new Tuple<int, int>(i, tmp)); //--------------dzielenie zadań na osobne wątki
            }
        }
    }

    internal sealed class RungeKutty1 : RungeKutty
    {
        //--Konstruktor----///
        public RungeKutty1()
        {
            Initialize();
        }
        private Func<int, double[], double> Function { get; set; }
        private ManualResetEvent DoneEvent { get; set; }
        //--------Runge-Kutty Pierwszego rzedu------------//
        private void Odeint_Euler(Func<int, double[], double> function)
        {
            Function = function;
            using (ManualResetEvent resetEvent = new ManualResetEvent(false))
            {
                for (int i = 0; i < Threads; ++i)
                {
                    ThreadPool.QueueUserWorkItem(ThreadPoolCallback, i);
                    resetEvent.Set();
                }
                resetEvent.WaitOne();
            }
        }

        private void InnerLoop(int index)
        {
            for(int i=Limiter[index].Item1; i<Limiter[index].Item2; ++i)
            {
                Ynast[i] = Y[i] + H * Function(i, Y);
            }
        }
        //-------Funkcja Wykonująca oraz zapisująca do pliku----//
        public override void Calculate()
        {
            using (var outputFile =
                new StreamWriter(@"C:\Users\Kevin\Documents\Visual Studio Projects\PPNVS\RungeKutty\RungeKutty1.dat"))
            {
                for (double t = 0; t < Tmax; t += H)
                {
                    Odeint_Euler(Osc_Spr);                   
                    outputFile.Write(t.ToString("F6"));
                    for (var i = 0; i < N; ++i)
                        outputFile.Write("\t" + Y[i].ToString("F6"));
                    outputFile.Write("\t" + H.ToString("F6"));
                    outputFile.Write("\n");

                    //zmiana tablic
                    var ytemp = Y;
                    Y = Ynast;
                    Ynast = ytemp;
                }
            }
        }
        //-------------Wrapper do korzystania Threadpool'a --------------------//  
        public void ThreadPoolCallback(object threadContext)
        {
            var threadIndex = (int)threadContext;
            InnerLoop(threadIndex);
        }
    }

    internal sealed class RungeKutty2 : RungeKutty
    {
        //-------Konstruktor-------------//
        public RungeKutty2()
        {
            Initialize();
        }
        //private double k1 = 0, k2 = 0;
        private Func<int, double[], double> Function { get; set; }
        private double[] ytemp = new double[N];
        private ManualResetEvent DoneEvent { get; set; }
        //------------Runge-Kutty Drugiego Rzędu-----------//
        private void InnerLoop(int index)
        {
            var ytemp = new double[N];
            for (var i = Limiter[index].Item1; i < Limiter[index].Item2; ++i)
            {
                double k1 = H * Function(i, Y);
                ytemp[i] = Y[i] + 0.5 * k1;
            }
            for (var i = Limiter[index].Item1; i < Limiter[index].Item2; ++i)
            {
                double k2 = H * Function(i, ytemp);
                Ynast[i] = Y[i] + k2;
            }
        }

        private void Odeint_Midpoint(Func<int, double[], double> function)
        {
            Function = function;
            using (ManualResetEvent resetEvent = new ManualResetEvent(false))
            {
                for (int i = 0; i < Threads; ++i)
                {
                    ThreadPool.QueueUserWorkItem(ThreadPoolCallback, i);
                    resetEvent.Set();
                }
                resetEvent.WaitOne();
            }         
        }

        //------------Funkcja Wykonująca oraz zapisująca do pliku----//
        public override void Calculate()
        {
            //Y[1] = 1; // w wypadku oscylatora
            using (var outputFile =
                new StreamWriter(@"C:\Users\Kevin\Documents\Visual Studio Projects\PPNVS\RungeKutty\RungeKutty2.dat"))
            {
                for (double t = 0; t < Tmax; t += H)
                {
                    Odeint_Midpoint(Osc_Spr);
                    outputFile.Write(t.ToString("F6"));
                    for (var i = 0; i < N; ++i)
                        outputFile.Write("\t" + Y[i].ToString("F6"));
                    outputFile.Write("\t" + H.ToString("F6"));
                    outputFile.Write("\n");

                    //zmiana tablic
                    var ytemp = Y;
                    Y = Ynast;
                    Ynast = ytemp;
                }
            }
        }

        //-------------Wrapper do korzystania Threadpool'a --------------------//  
        public void ThreadPoolCallback(Object threadContext)
        {
            var threadIndex = (int)threadContext;
            InnerLoop(threadIndex);
        }
    }

    internal sealed class RungeKutty4 : RungeKutty
    {
        public RungeKutty4()
        {
            Initialize();
        }
        private Func<int, double[], double> Function { get; set; }
        private double[] ytemp = new double[N];
        public override void Calculate()
        {
            using (var outputFile =
                new StreamWriter(@"C:\Users\Kevin\Documents\Visual Studio Projects\PPNVS\RungeKutty\RungeKutty3.dat"))
            {
                Y[1] = 1;
                Y[14 - 1] = -1;
                for (double t = 0; t < Tmax; t += H)
                {
                    Odeint_Rk4(Osc_Spr);
                    outputFile.Write(t.ToString("F6"));
                    for (var i = 0; i < N; ++i)
                        outputFile.Write("\t" + Y[i].ToString("F6"));
                    outputFile.Write("\t" + H.ToString("F6"));
                    outputFile.Write("\n");

                    //zmiana tablic
                    var ytemp = Y;
                    Y = Ynast;
                    Ynast = ytemp;
                }
            }
        }

        private void InnerLoop(int index)
        {
            double[] ytemp = new double[N];
            double[] k1 = new double[N];
            double[] k2 = new double[N];
            double[] k3 = new double[N];
            double[] k4 = new double[N];
            for (var i = Limiter[index].Item1; i < Limiter[index].Item2; ++i)
            {
                k1[i] = H * Function(i, Y);
                ytemp[i] = Y[i] + 0.5 * k1[i];
            }
            for (var i = Limiter[index].Item1; i < Limiter[index].Item2; ++i)
            {
                k2[i] = H * Function(i, ytemp);
                Ynast[i] = Y[i] + 0.5 * k2[i];
            }
            for (var i = Limiter[index].Item1; i < Limiter[index].Item2; ++i)
            {
                k3[i] = H * Function(i, Ynast);
                ytemp[i] = Y[i] + k3[i];
            }
            for (var i = Limiter[index].Item1; i < Limiter[index].Item2; ++i)
            {
                k4[i] = H * Function(i, ytemp);
                Ynast[i] = Y[i] + (k1[i] + 2.0 * k2[i] + 2.0 * k3[i] + k4[i]) / 6.0;
            }
        }

        public void Odeint_Rk4(Func<int, double[], double> function)
        {
            Function = function;
            using (ManualResetEvent resetEvent = new ManualResetEvent(false))
            {
                for (int i = 0; i < Threads; ++i)
                {
                    ThreadPool.QueueUserWorkItem(ThreadPoolCallback, i);
                    resetEvent.Set();
                }
                resetEvent.WaitOne();
            }
        }
        public void ThreadPoolCallback(Object threadContext)
        {
            var threadIndex = (int)threadContext;
            InnerLoop(threadIndex);
        }
    }

    internal class Program
    {
        public static void Main(string[] args)
        {
            Stopwatch stopwatch = new Stopwatch();
            var rk1 = new RungeKutty1();
            var rk2 = new RungeKutty2();
            var rk4 = new RungeKutty4();

            stopwatch.Start();
            //rk1.Calculate();
            //rk2.Calculate();
            rk4.Calculate();
            stopwatch.Stop();
            
            TimeSpan timespan = stopwatch.Elapsed;
            string elapsedTimeWithoutThreads =
                $"{timespan.Hours:00}:{timespan.Minutes:00}:{timespan.Seconds:00}.{timespan.Milliseconds / 10:00}";
            stopwatch.Reset();

            Console.WriteLine("Time: " + elapsedTimeWithoutThreads);
            Console.ReadKey();
            
        }
    }
}